' RockScissorsPaperPath Puzzle for CMM2' Rev 1.0.0 William M Leue 4-Aug-2022

option default integer
option base 1

const NCOLS = 4
const NROWS = 6
const NCELLS = NCOLS*NROWS
const STACKSIZE = 200
const NCOMPS = 4

const CX_ROW = 1
const CX_COL = 2
const CX_DIR = 3
const CX_NM  = 4

const EMPTY   = 0
const VISITED = -1
const MAX_ITER  = 25

const NORTH = 1
const SOUTH = 2
const EAST  = 3
const WEST  = 4
const NDIRS = 4

const ROCK     = 1
const SCISSORS = 2
const PAPER    = 3
const NICONS   = 3

const LINK_OFF = 0
const LINK_ON  = 1
const LINK_DEL = 2
const HORIZ = 1
const VERT = 2

' Graphic Constants
const GNCOLS = 7
const GNROWS = 11
const CSIZE = 40
const BW = GNCOLS*CSIZE
const BH = GNROWS*CSIZE
const BX = mm.hres\2 - BW\2
const BY = mm.vres\2 - BH\2
const MWIDTH = 20
const CRAD = 5
const LWIDTH = 3

const BGCOLOR = rgb(180, 180, 180)
const ALCOLOR = rgb(white)

const XL    = CSIZE\3
const XW    = 2
const ASW   = 2
const ASL   = CSIZE\2-1
const ASR   = 5
const AHH   = XW + 6

const MCHAN = 2

const ESC    = 27
const ENTER  = 13
const HOME   = 134
const F1     = 145
const QUEST  = 63
const INSERT = 132

const MAX_HINTS = 1
const LINK_MISSING = 0
const LINK_OK      = 1
const LINK_EXTRA   = 2
const LINK_IMPOSS  = 3

' Globals
dim board(NROWS, NCOLS)
dim icons(NROWS, NCOLS)
dim path(3, NCELLS+1)
dim links(GNROWS, GNCOLS)
dim linkmap(10, NCELLS)
dim stack(NCOMPS, STACKSIZE)
dim errors(GNROWS, GNCOLS)
dim sp = 0
dim current_row = 0
dim current_col = 0
dim start_row = 0
dim start_col = 0
dim directions(NDIRS) = (NORTH, SOUTH, EAST, WEST)
dim rock_vecs(8, 2)
dim scissor_vecs1(5, 2)
dim scissor_vecs2(5, 2)
dim paper_vecs(5, 2)
dim mouse_x = 0
dim mouse_y = 0
dim left_click = 0
dim left_busy = 0
dim lp = 0
dim count = 0
dim iter = 0
dim cheat = 0
dim nhints = 0
dim running = 0

' Main Program
open "debug.txt" for output as #1
mode 1, 16
ReadIconData
ReadLinkMap
InitMouse
NewGame
HelpNeeded
DrawBoard
DrawIcons
HandleEvents
end

' Read Icon vector data
sub ReadIconData
  local i, j
  for i = 1 to 8
    for j = 1 to 2
      read rock_vecs(i, j)
    next j
  next i
  for i = 1 to 5
    for j = 1 to 2
      read scissor_vecs1(i, j)
    next j
  next i
  for i = 1 to 5
    for j = 1 to 2
      read scissor_vecs2(i, j)
    next j
  next i
  for i = 1 to 5
    for j = 1 to 2
      read paper_vecs(i, j)
    next j
  next i
end sub

' Read the data for mapping path coordinates and directions
' to Link graphic coordinates.
sub ReadLinkMap
  local i, j
  for i = 1 to NCELLS
    for j = 1 to 10
      read linkmap(j, i)
    next j
  next i
end sub

' Initialize for a new game
sub NewGame
  local row, col, i
  cls
  text mm.hres\2, mm.vres\2, "Creating New Game...", "CM", 4
  current_row = 0
  current_col = 0
  lp = 1
  ClearStack
  ClearLinks
  FindPath
  PlaceIcons
  nhints = 0
  running = 1
end sub

' Find a non-intersecting path that goes through all NCELLS cells on the board.
' All links on the path are Manhattan arcs. This uses a very simple recursive
' mechanism and multiple trials. Success usually happens in less than 10 seconds,
' but sometimes 2 or 3 times slower.
sub FindPath
  local tcount = 0
  do
    ClearBoard
    PickStartCell
    count = 1
    lp = 1
    sp = 1
    path(1, lp) = current_row : path(2, lp) = current_col : path(3, lp) = 0
    current_row = start_row
    current_col = start_col
    max_count = 0
    MakeTrialPath
    inc tcount
  loop until count = NCELLS
end sub

' Just try to find adjacent cells until we get blocked. Picks random directions, backtracks
' if cannot find a non-visted cell, and gives up if there are no more pushed cells
' to retry or we run out of iterations. This is simpler and faster than a full
' backtracking.
sub MakeTrialPath
  local row, col, nmoves, nrow, ncol, dir
  local moves(2, NDIRS)
  row = current_row : col = current_col
  iter = 0
  do
    inc iter
    if iter > MAX_ITER then
      exit sub
    end if
    GetMoves row, col, nmoves, moves()
    if nmoves > 0 then
      mx = ChooseMove(nmoves, moves())
      nrow = moves(CX_ROW, mx)
      ncol = moves(CX_COL, mx)
      PushMove row, col, mx, nmoves-1
      inc count
      row = nrow : col = ncol
      board(row, col) = VISITED
      current_row = row : current_col = col
      path(3, lp) = mx
      inc lp
      path(1, lp) = current_row : path(2, lp) = current_col
      if count >= NCELLS then exit sub
    else
      do
        PopMove prow, pcol, dir, nmoves
        if prow = 0 then exit sub
        GetCoords prow, pcol, dir, nrow, ncol
        board(nrow, ncol) = EMPTY
        inc count, -1
        row = prow : col = pcol
        current_row = row : current_col = col
        inc lp, -1
      loop until nmoves > 0
    end if
  loop until count >= NCELLS
end sub

' From the current location, make a list of the available moves
' Up to 4 moves can exist (Moore Neighborhood) but some are invalid
' due to being off the board or to previously visited cells.
sub GetMoves row, col, nmoves, moves()
  local nrow, ncol, dir
  nmoves = 0
  for dir = 1 to 4
    moves(CX_ROW, dir) = 0
    moves(CX_COL, dir) = 0
    GetCoords row, col, dir, nrow, ncol
    if nrow < 1 or nrow > NROWS then continue for
    if ncol < 1 or ncol > NCOLS then continue for
    if board(nrow, ncol) <> EMPTY then continue for
    inc nmoves
    moves(CX_ROW, dir) = nrow
    moves(CX_COL, dir) = ncol
  next dir
end sub

' Choose a move out of the available valid ones
function ChooseMove(nmoves, moves())
  local m
  do
    m = RandomIntegerInRange(1, NDIRS)
    if moves(1, m) > 0 then exit do
  loop
  ChooseMove = m
end function

' Given a cell and a direction, compute the coordinates of a possible neigbbor
' Note that the coordinates may be off the board!
sub GetCoords row, col, dir, nrow, ncol
  select case dir
    case 1
      nrow = row-1 : ncol = col
    case 2
      nrow = row+1 : ncol = col
    case 3
      nrow = row   : ncol = col+1
    case 4
      nrow = row   : ncol = col-1
  end select
end sub

' Clear the Board
sub ClearBoard
  local row, col, pp
  for row = 1 to NROWS
    for col = 1 to NCOLS
      board(row, col) = EMPTY
    next col
  next row
end sub      

' Clear the Backtrack Stack
sub ClearStack
  sp = 0
end sub

' Push a move onto the Backtrack stack
sub PushMove row, col, dir, nmoves
  inc sp
  stack(CX_ROW, sp) = row
  stack(CX_COL, sp) = col
  stack(CX_DIR, sp) = dir
  stack(CX_NM,  sp) = nmoves
end sub

' Pop a move off the Backtrack stack
sub PopMove row, col, dir, nmoves
  if sp < 2 then
    row = 0 : col = 0 : dir = 0 : nmoves = 0
    exit sub
  end if
  row = stack(CX_ROW, sp)
  col = stack(CX_COL, sp)
  dir = stack(CX_DIR, sp)
  nmoves = stack(CX_NM, sp)
  inc sp, -1
end sub

' Pick a random starting cell
sub PickStartCell
  start_col = RandomIntegerInRange(1, NCOLS)
  start_row = RandomIntegerInRange(1, NROWS)
  current_col = start_col
  current_row = start_row
  board(current_row, current_col) = 1
end sub

' Place the icons at the board nodes, based on the path
sub PlaceIcons
  local prow, pcol, t, x1, y1, x2, y2
  local row, col, srow, scol, flag, i, ix
  ix = ROCK
  x1 = 0 : y1 = 0
  for i = 1 to NCELLS
    prow = path(1, i)
    pcol = path(2, i)
    icons(prow, pcol) = ix
    row = prow*2-1
    col = pcol*2-1
    x2 = BX + (col-1)*CSIZE + CSIZE\2
    y2 = BY + (row-1)*CSIZE + CSIZE\2
    inc ix
    if ix > NICONS then ix = ROCK
    x1 = x2 : y1 = y2
  next i
end sub

' Init mouse and cursor
sub InitMouse
  on error skip 1
  controller mouse open MCHAN, LClick
  if mm.errno <> 0 then
    cls
    print "Could not open mouse on channel " + str$(MCHAN)
    print "Press any key to exit"
    z$ = INKEY$
    do
      z$ = INKEY$
    loop until z$ <> "'
    cls
    end
  end if
  gui cursor on 1, mm.hres\2, mm.vres\2, rgb(red)
  settick 20, UpdateCursor
end sub

' Process Left Mouse Click
sub LClick
  if left_busy = 0 then
    mouse_x = mouse(X)
    mouse_y = mouse(Y)
    left_click = 1
  end if
end sub

' Make cursor track the mouse
sub UpdateCursor
  gui cursor mouse(X), mouse(Y)
end sub

' Draw the playing board
sub DrawBoard
  local row, col, x, y, brow, bcol, c, h
  cls
  gui cursor hide
  box BX-MWIDTH, BY-MWIDTH, BW+2*MWIDTH, BH+2*MWIDTH,, BGCOLOR, BGCOLOR
  for row = 1 to GNROWS
    y = BY + (row-1)*CSIZE
    for col = 1 to GNCOLS
      x = BX + (col-1)*CSIZE
      h = 0
      if (row mod 2 = 1) and (col mod 2) = 0 then
        c = ALCOLOR
        h = 1
       else if (row mod 2 = 0) and (col mod 2) = 1 then
        c = ALCOLOR
        h = 1
      else
        c = BGCOLOR
      end if
      box x, y, CSIZE, CSIZE,, c, c
      if h and links(row, col) = LINK_OFF then
        circle x+CSIZE\2, y+CSIZE\2, CRAD,,, BGCOLOR, BGCOLOR
      end if
      if (row mod 2) = 1 and (col mod 2) = 1 then
        brow = (row+1)\2
        bcol = (col+1)\2
        if brow = start_row and bcol = start_col then
            c = rgb(yellow)
        else
            c = BGCOLOR
        end if
      else
        if links(row, col) <> LINK_OFF then
          dir = GetLinkDirection(row, col)
          DrawLink row, col, dir
        end if
      end if
    next col
  next row
  gui cursor show
  text mm.hres\2, 10, "Rock-Scissors-Paper-PATH", "CT", 4,, rgb(green)
end sub

' Draw the Icons
sub DrawIcons
  local prow, pcol, t, x1, y1, x2, y2
  local row, col, srow, scol, flag, i, ix
  x1 = 0 : y1 = 0
  for i = 1 to NCELLS
    prow = path(1, i)
    pcol = path(2, i)
    ix = icons(prow, pcol)
    row = prow*2-1
    col = pcol*2-1
    x2 = BX + (col-1)*CSIZE + CSIZE\2
    y2 = BY + (row-1)*CSIZE + CSIZE\2
    if cheat then
      if x1 > 0 then
        line x1, y1, x2, y2,, rgb(green)
      end if
    end if
    DrawIcon x2, y2, ix
    x1 = x2 : y1 = y2
  next i
end sub

' Draw the specified icon at the specified location
sub DrawIcon x, y, ix
  local xv(10), yv(10), i, cx, cy
  cx = x
  cy = y
  select case ix
    case ROCK
      for i = 1 to 8
        xv(i) = cx - CSIZE\2 + rock_vecs(i, 1)
        yv(i) = cy - CSIZE\2 + rock_vecs(i, 2)
      next i
      polygon 8, xv(), yv(), rgb(black), BGCOLOR
      line xv(1), yv(1), xv(1)-7, yv(1)+17,, rgb(black)
      line xv(1), yv(1), xv(1)+8, yv(1)+20,, rgb(black)
    case SCISSORS
      circle x-10, y+CSIZE\2-11, 5,,, rgb(black), rgb(black)
      circle x-10, y+CSIZE\2-11, 4,,, rgb(black), BGCOLOR
      circle x+10, y+CSIZE\2-11, 5,,, rgb(black), rgb(black)
      circle x+10, y+CSIZE\2-11, 4,,, rgb(black), BGCOLOR
      for i = 1 to 5
        xv(i) = cx + scissor_vecs1(i, 1)
        yv(i) = cy + scissor_vecs1(i, 2)
      next i
      polygon 5, xv(), yv(), rgb(black), rgb(black)
      for i = 1 to 5
        xv(i) = cx + scissor_vecs2(i, 1)
        yv(i) = cy + scissor_vecs2(i, 2)
      next i
      polygon 5, xv(), yv(), rgb(black), rgb(black)
    case PAPER
      for i = 1 to 5
        xv(i) = cx - CSIZE\2 + paper_vecs(i, 1)
        yv(i) = cy - CSIZE\2 + paper_vecs(i, 2)
      next i
      polygon 5, xv(), yv(), rgb(black), rgb(white)
      line xv(2), yv(2), xv(2), yv(3),, rgb(black)
      line xv(2), yv(3), xv(3), yv(3),, rgb(black)
  end select
end sub

' Clear the Links
sub ClearLinks
  local row, col
  for row = 1 to GNROWS
    for col = 1 to GNCOLS
      links(row, col) = LINK_OFF
    next col
  next row
end sub

' Get the link direction (N, S, E, W) for a specified cell
' by looking at the icons on each side.
function GetLinkDirection(grow, gcol)
  local r1, c1, r2, c2, ldir
  if (grow mod 2) = 0 then dir = VERT else dir = HORIZ
  if dir = VERT then
    r1 = (grow-1)\2 + 1
    c1 = gcol\2+1
    icon1 = icons(r1, c1)
    r2 = (grow+1)\2 + 1
    c2 = gcol\2+1
    icon2 = icons(r2, c2)
    if icon1 = ROCK then
      if icon2 = SCISSORS then
        ldir = SOUTH
      else if icon2 = PAPER then
        ldir = NORTH
      else
        ldir = 0
      end if
    else if icon1 = SCISSORS then
      if icon2 = PAPER then
        ldir = SOUTH
      else if icon2 = ROCK then
        ldir = NORTH
      else
        ldir = 0
      end if
    else
      if icon2 = ROCK then
        ldir = SOUTH
      else if icon2 = SCISSORS then
        ldir = NORTH
      else
        ldir = 0
      end if
    end if
  else
    r1 = grow\2 + 1
    c1 = (gcol-1)\2+1
    icon1 = icons(r1, c1)
    r2 = grow\2 + 1
    c2 = (gcol+1)\2+1
    icon2 = icons(r2, c2)
    if icon1 = ROCK then
      if icon2 = SCISSORS then
        ldir = EAST
      else if icon2 = PAPER then
        ldir = WEST
      else
        ldir = 0
      end if
    else if icon1 = SCISSORS then
      if icon2 = PAPER then
        ldir = EAST
      else if icon2 = ROCK then
        ldir = WEST
      else
        ldir = 0
      end if
    else
      if icon2 = ROCK then
        ldir = EAST
      else if icon2 = SCISSORS then
        ldir = WEST
      else
        ldir = 0
      end if
    end if
  end if
  GetLinkDirection = ldir
end function

' Draw a Link in one of 3 possible states:
' LINK_OFF (nothing drawn)
' LINK_ON  (yellow arrow)
' LINK_DEL (red X on top of yellow arrow)
' Direction of link = dir (N, S, E, W)
sub DrawLink grow, gcol, ldir
  local lnk, x1, y1, x2, y2, dir, xc, yc
  local xv(12), yv(12), xt, yt
  lnk = links(grow, gcol)
  if lnk = LINK_OFF then exit sub
  xc = BX + (gcol-1)*CSIZE + CSIZE\2
  yc = BY + (grow-1)*CSIZE + CSIZE\2
  if (grow mod 2) = 0 then dir = VERT else dir = HORIZ
  select case ldir
    case NORTH
      xv(1) = xc + ASW : yv(1) = yc + ASL
      xv(2) = xc + ASW : yv(2) = yc - ASR
      xv(3) = xc + AHH : yv(3) = yc - ASR
      xv(4) = xc       : yv(4) = yc - ASL
      xv(5) = xc - AHH : yv(5) = yc - ASR
      xv(6) = xc - ASW : yv(6) = yc - ASR
      xv(7) = xc - ASW : yv(7) = yc + ASL
      polygon 7, xv(), yv(), rgb(black), rgb(yellow)
    case SOUTH
      xv(1) = xc + ASW : yv(1) = yc - ASL
      xv(2) = xc + ASW : yv(2) = yc + ASR
      xv(3) = xc + AHH : yv(3) = yc + ASR
      xv(4) = xc       : yv(4) = yc + ASL
      xv(5) = xc - AHH : yv(5) = yc + ASR
      xv(6) = xc - ASW : yv(6) = yc + ASR
      xv(7) = xc - ASW : yv(7) = yc - ASL
      polygon 7, xv(), yv(), rgb(black), rgb(yellow)
    case EAST
      xv(1) = xc - ASL : yv(1) = yc - ASW
      xv(2) = xc + ASR : yv(2) = yc - ASW
      xv(3) = xc + ASR : yv(3) = yc - AHH 
      xv(4) = xc + ASL : yv(4) = yc
      xv(5) = xc + ASR : yv(5) = yc + AHH
      xv(6) = xc + ASR : yv(6) = yc + ASW
      xv(7) = xc - ASL : yv(7) = yc + ASW
      polygon 7, xv(), yv(), rgb(black), rgb(yellow)
    case WEST
      xv(1) = xc + ASL : yv(1) = yc - ASW
      xv(2) = xc - ASR : yv(2) = yc - ASW
      xv(3) = xc - ASR : yv(3) = yc - AHH 
      xv(4) = xc - ASL : yv(4) = yc
      xv(5) = xc - ASR : yv(5) = yc + AHH
      xv(6) = xc - ASR : yv(6) = yc + ASW
      xv(7) = xc + ASL : yv(7) = yc + ASW
      polygon 7, xv(), yv(), rgb(black), rgb(yellow)
    case default
  end select  
  if lnk = LINK_DEL then
    xt = xc - 0.707*XL : yt = yc - 0.707*XL
    x1 = xt + 0.707*XW : y1 = yt - 0.707*XW
    x2 = xt - 0.707*XW : y2 = yt + 0.707*XW
    xv(1) = x1             : yv(1) = y1
    xv(12) = x2            : yv(12) = y2
    xv(2) = xc             : yv(2) = yc - XW
    xt = xc + 0.707*XL : yt = yc - 0.707*XL
    x1 = xt - 0.707*XW : y1 = yt - 0.707*XW
    x2 = xt + 0.707*XW : y2 = yt + 0.707*XW
    xv(3) = x1             : yv(3) = y1
    xv(4) = x2             : yv(4) = y2
    xv(5) = xc + XW  : yv(5) = yc
    xt = xc + 0.707*XL : yt = yc + 0.707*XL
    x1 = xt + 0.707*XW : y1 = yt - 0.707*XW
    x2 = xt - 0.707*XW : y2 = yt + 0.707*XW
    xv(6) = x1             : yv(6) = y1
    xv(7) = x2             : yv(7) = y2
    xv(8) = xc             : yv(8) = yc + XW
    xt = xc - 0.707*XL : yt = yc + 0.707*XL
    x1 = xt + 0.707*XW : y1 = yt + 0.707*XW
    x2 = xt - 0.707*XW : y2 = yt - 0.707*XW
    xv(9) = x1             : yv(9) = y1
    xv(10) = x2            : yv(10) = y2
    xv(11) = xc - XW : yv(11) = yc
    polygon 12, xv(), yv(), rgb(red), rgb(red)
  end if
end sub

' Handle User Events
sub HandleEvents
  local z$, cmd, w
  z$ = INKEY$
  do
    if left_click then
      left_busy = 1
      HandleLeftClick
      left_click = 0
      left_busy = 0
    end if
    z$ = INKEY$
    if z$ <> "" then
      cmd = asc(UCASE$(z$))
      select case cmd
        case ESC
          Quit
        case ENTER
          cheat = 1 - cheat
          DrawBoard
          DrawIcons
        case HOME
          ClearLinks
          DrawBoard
          DrawIcons
        case F1
          NewGame
          DrawBoard
          DrawIcons
        case INSERT
          if running then
            ShowHint
          end if
        case QUEST
          ShowHelpScreen
      end select
    end if
  loop
end sub

' Process left mouse click
sub HandleLeftClick
  local grow, gcol
  if not running then exit sub
  if mouse_x < BX or mouse_x > BX+BW then exit sub
  if mouse_y < BY or mouse_y > BY+BH then exit sub
  grow = (mouse_y - BY)\CSIZE + 1
  gcol = (mouse_x - BX)\CSIZE + 1
  if (grow mod 2) = 1 then
    if (gcol mod 2) = 0 then
      BumpLink grow, gcol
      DrawBoard
      DrawIcons
      if IsWon() then ShowWin
    end if
  else if (grow mod 2) = 0 then
    if (gcol mod 2) = 1 then
      BumpLink grow, gcol
      DrawBoard      
      DrawIcons
      if IsWon() then ShowWin
    end if
  end if
end sub

' Bunp a selected link to next state if possible
sub BumpLink grow, gcol
  local dir, gr1, gc1, gr2, gc2, icon1, icon2
  local r1, c1, r2, c2
  if (grow mod 2) = 0 then dir = VERT else dir = HORIZ
  if dir = vert then
    gr1 = grow-1 : gc1 = gcol
    gr2 = grow+1 : gc2 = gcol
  else
    gr1 = grow : gc1 = gcol-1
    gr2 = grow : gc2 = gcol+1
  end if
  r1 = gr1\2+1 : c1 = gc1\2+1
  r2 = gr2\2+1 : c2 = gc2\2+1
  icon1 = icons(r1, c1)
  icon2 = icons(r2, c2)
  if icon2 = icon1 then exit sub
  inc links(grow, gcol)
  if links(grow, gcol) > LINK_DEL then
    links(grow, gcol) = LINK_OFF
  end if
end sub

' Detect a won game
' Criteria for a win:
'  1. All Nodes must have a corresponding link
'  2. There must be no extraneous links, including cancelled links.
function isWon()
  local i, row, col, dir, grow, gcol, nmatch, nlinks
  nmatch = 0
  nlinks = 0
  for i = 1 to NCELLS
    row = path(1, i)
    col = path(2, i)
    dir = path(3, i)
    if dir > 0 then
    FindLinkCoords row, col, dir, grow, gcol
      if links(grow, gcol) = LINK_ON then
        inc nmatch
      end if        
    end if
  next i
  IsWon = (nmatch = NCELLS-1)
  for grow = 1 to GNROWS
    for gcol = 1 to GNCOLS
      if links(grow, gcol) <> LINK_NO then
        inc nlinks
      end if
    next gcol
  next grow
  if nlinks > NCELLS-1 then IsWon = 0 
end function

' End the game when it is won and show the win
sub ShowWin
  running = 0
  text mm.hres\2, 10, space$(20), "CT", 5,, rgb(black)
  text mm.hres\2, 10, "You Win!", "CT", 5,, rgb(green)
end sub

' Using the LinkMap data, find the link coordinates that correspond to
' a move along the path from cell (row, col) in direction 'dir'.
sub FindLinkCoords row, col, dir, grow, gcol
  local i, j
  for i = 1 to NCELLS
    if linkmap(1, i) = row and linkmap(2, i) = col then
      j = dir*2+1
      grow = linkmap(j, i)
      gcol = linkmap(j+1, i)
      exit sub
    end if
  next i
end sub

' Get a hint unless you've already used yours up
' This will only show a missing link, not an exraneous one
sub ShowHint
  local i, row, col, dir, grow, gcol, x, y
  if nhints >= MAX_HINTS then exit sub
  inc nhints
  do
    i = RandomIntegerInRange(1, lp-1)
    row = path(1, i)
    col = path(2, i)
    dir = path(3, i)
    FindLinkCoords row, col, dir, grow, gcol
    if links(grow, gcol) <> LINK_ON then exit do
  loop
  x = BX + (gcol-1)*CSIZE
  y = BY + (grow-1)*CSIZE
  box x, y, CSIZE, CSIZE, 2, rgb(red)
  pause 2000
  DrawBoard
  DrawIcons
end sub

' Ask if help needed
sub HelpNeeded
  local z$, cmd
  cls
  text mm.hres\2, 10, "Rock-Scissors-Paper-PATH Game", "CT", 5,, rgb(green)
  print @(0, 100) "Need Help? -- Press '?' or any other key to play."
  do
    z$ = INKEY$
  loop until z$ <> ""
  cmd = asc(UCASE$(z$))
  if cmd = QUEST then
    ShowHelpScreen
  else
    cls
    DrawBoard
    DrawIcons
  end if
end sub

' Help Screen
sub ShowHelpScreen
  local x, y, xf, yf
  cls
  running = 0
  text mm.hres\2, 10, "Rock-Scissors-Paper-PATH Rules", "CT", 4,, rgb(green)
  print @(0, 40) ""
  print "Everyone knows the rules for the Rock-Scissors-Paper game, right?"
  print "Rock breaks Scissors, Scissors cut Paper, Paper wraps Rock, or in pictures:"

  x = 200 : y = 98 : xf = 30 : yf = 0
  box 175, 80, 355, 37,, BGCOLOR, BGCOLOR
  DrawIcon x, y, ROCK
  DrawArrow x+xf, y+yf
  inc x, 100
  DrawIcon x, y, SCISSORS
  DrawArrow x+xf, y+yf
  inc x, 100
  DrawIcon x, y, PAPER 
  DrawArrow x+xf, y+yf
  inc x, 100
  DrawIcon x, y, ROCK

  print @(0, 108) ""
  print "The Rock-Scissors-Paper-PATH game takes the old game one step farther:"
  print "You are presented with a grid of Rock-Scissors-Paper icons, and you must find"
  print "a path that connects all the 24 icons into a single path using only horizontal"
  print "and vertical links between neighboring icons. The path must not intersect itself,"
  print "and each icon must be visited exactly once. The path is always open: that is, "
  print "the start and end icons are always different. The correct path is created"
  print "randomly for each game but hidden from you: your only clue is the arrangement of icons."
  print "Although there are typically several paths that match the visible icons, only the"
  print "links that exactly match the hidden path will win the puzzle."
  print ""
  print "To make a link, click the white circle between two icons (horizontally or vertically),"
  print "and a yellow arrow will be formed. Note that each link MUST honor the rules of"
  print "the original Rock-Scissors-Paper game: that is, the link must go from a Rock icon"
  print "to a Scissors icon, a Scissors icon to a Paper icon, or a Paper icon to a Rock icon."
  print "The game will prevent you from making a silly link, such Paper -> Paper."
  print ""
  print "To delete a link, click it again. The link will be decorated with a red X, showing"
  print "your intent to delete it, but leaving it as a help to remember you tried making that"
  print "link before. To really delete it, click the link again.
  print ""
  print "To win the game, you must complete the path AND you cannot have any extraneous links"
  print "that don't belong to the path, including links marked with a red X."
  print ""
  print "If you get stuck, you can get a hint by pressing the Insert Key on the keyboard. This"
  print "will highlight in RED one link that is missing. The highlight will last for 2 seconds."
  print "The hints will NOT highlight spurious links. But be sparing in your use of this"
  print "hint: you get a maximum of 1 hint per game!"
  print ""
  print "Keyboard Commands are:
  print "  Restart the game:    Home Key
  print "  New Game:            F1 Key
  print "  Help:                ? Key
  print "  Hint:                Insert Key
  print "  Quit:                Escape Key
  print ""
  print "Press Any Key to Show a Sample WonGame with the links correctly aligned with the path."
  print "The starting icon is outlined in green; the ending icon is outlined in red."
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls
  PrintWonGame
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  NewGame
  DrawBoard
  DrawIcons
end sub  

' Draw a right-facing arrow at the coords
sub DrawArrow x, y
  local i
  line x, y, x+40, y, 3, rgb(yellow)
  for i = 0 to 2
    line x+40+i, y+1, x+35+i, y-5, 3, rgb(yellow)
    line x+40+i, y+1, x+34+i, y+7, 3, rgb(yellow)
  next i
end sub

' Quit the Program
sub Quit
  settick 0, 0
  gui cursor off
  controller mouse close
  cls
  end
end sub

' Generate a uniformly distributed random integer in the closed range a to b
' (gets around issues with non-uniform distribution in rnd() at some
' expense in performance.)
function RandomIntegerInRange(a as integer, b as integer) as integer
  local v, c
  c = b - a + 1
  do
    v = a + (b-a+2)*rnd()
    if v > 1 and v-a <= c then exit do
  loop
  RandomIntegerInRange = v-1
end function

' Print a sample Won game for the Help Screen
sub PrintWonGame
  local i, row, col, lrow, lcol, dir, nrow, ncol, grow, gcol
  local x, y
  for i = 1 to lp-1
    row = path(1, i)
    col = path(2, i)
    dir = path(3, i)
    FindLinkCoords row, col, dir, lrow, lcol
    links(lrow, lcol) = LINK_ON
  next i
  DrawBoard
  DrawIcons
  row = path(1, 1) : col = path(2, 1)
  grow = row*2 - 1 : gcol = col*2 - 1     
  x = BX + (gcol-1)*CSIZE
  y = BY + (grow-1)*CSIZE
  box x, y, CSIZE, CSIZE, 2, rgb(green)
  row = path(1, lp) : col = path(2, lp)
  grow = row*2 - 1 : gcol = col*2 - 1     
  x = BX + (gcol-1)*CSIZE
  y = BY + (grow-1)*CSIZE
  box x, y, CSIZE, CSIZE, 2, rgb(red)
  text mm.hres\2, 10, "Sample Won Game", "CT", 5,, rgb(green)
end sub

' (debug)
sub PrintErrors
  local grow, gcol
  for grow = 1 to GNROWS
    for gcol = 1 to GNCOLS
      print #1, str$(errors(grow, gcol)) + " ";
    next gcol
    print #1, ""
  next grow
end sub

' Polygon coordinates for Icons

' Rock
data 20,6,  34,11,  36,29,  27,31,  17,27,  10,31,  3,26,  12, 11

' Scissors (blades)
data -9,9,  0,0,   10,-10,  0,3,  -6,6
data  9,9,  0,0,  -10,-10,  0,3,   6,6
 
' Paper 
data 9,5,  25,5,  31,11,  31,35,  9,35

' Path Coordinates to Link Coordinates mapping
' First pair: starting path coordinates
' Next 4 pairs: link coords for N, S, E, W
' (0, 0) means no path or link for that direction.
data 1, 1, 0, 0, 2, 1, 1, 2, 0, 0
data 1, 2, 0, 0, 2, 3, 1, 4, 1, 2
data 1, 3, 0, 0, 2, 5, 1, 6, 1, 4
data 1, 4, 0, 0, 2, 7, 0, 0, 1, 6
data 2, 1, 2, 1, 4, 1, 3, 2, 0, 0
data 2, 2, 2, 3, 4, 3, 3, 4, 3, 2
data 2, 3, 2, 5, 4, 5, 3, 6, 3, 4
data 2, 4, 2, 7, 4, 7, 0, 0, 3, 6
data 3, 1, 4, 1, 6, 1, 5, 2, 0, 0
data 3, 2, 4, 3, 6, 3, 5, 4, 5, 2
data 3, 3, 4, 5, 6, 5, 5, 6, 5, 4
data 3, 4, 4, 7, 6, 7, 0, 0, 5, 6
data 4, 1, 6, 1, 8, 1, 7, 2, 0, 0
data 4, 2, 6, 3, 8, 3, 7, 4, 7, 2
data 4, 3, 6, 5, 8, 5, 7, 6, 7, 4
data 4, 4, 6, 7, 8, 7, 0, 0, 7, 6
data 5, 1, 8, 1, 10,1, 9, 2, 0, 0
data 5, 2, 8, 3, 10,3, 9, 4, 9, 2
data 5, 3, 8, 5, 10,5, 9, 6, 9, 4
data 5, 4, 8, 7, 10,7, 0, 0, 9, 6
data 6, 1, 10,1, 0, 0, 11,2, 0, 0
data 6, 2, 10,3, 0, 0, 11,4, 11,2
data 6, 3, 10,5, 0, 0, 11,6, 11,4
data 6, 4, 10,7, 0, 0, 0, 0, 11,6

